<?php

namespace App\Http\Controllers\Admin;

use App\Models\User;
use App\Models\Deposit;
use Illuminate\Http\Request;
use App\Models\DepositMethod;
use App\Http\Controllers\Controller;
use App\Notifications\User\UserNotice;

class DepositController extends Controller {
    public function index() {
        
        $deposits = Deposit::latest()->paginate(10);
        return view('admin.deposits.all', [
            'deposits' => $deposits,
        ]);
    }

    public function single(Deposit $deposit) {
        $deposit_methods = DepositMethod::active()->get();
        return view('admin.deposits.single', compact('deposit', 'deposit_methods'));
    }

    public function update(Deposit $deposit, Request $request) {
        $data = $request->validate([
            'deposit_method_id' => 'required|numeric',
            'amount' => 'required|numeric',
            'comment' => 'nullable|string',
            'status' => 'required|string',
            'reject_reason' => 'nullable|string',
        ]);

        if($request->status == 'pending') {
            unset($data['status']);
        }

        if($deposit->status == 'approved' || $deposit->status == 'rejected') {
            return to_route('admin.deposits.all')->with('error', $this->message("You have already {$deposit->status} this deposit.", 'Approving or rejecting a deposit is a permanent action.'));
        }

        $deposit->update($data);
        $user = User::find($deposit->user->id);

        if ($request->status !== 'pending' && $request->send_email) {
            if ($request->status == 'approved') {
                $user->details()->increment('account_balance', $deposit->amount);
                
                $messages = [
                    "Your deposit of {$deposit->total_amount} has been approved and your account has been credited with $deposit->total_amount",
                    "Your total balance is {$user->total_balance}",
                ];
                $user->notify(new UserNotice(subject: "Deposit $request->status", messages: $messages));
            }

            if($request->status == 'rejected') {
                $messages = [
                    "Your deposit of {$deposit->total_amount} has been rejected due to the below reason:",
                    "$request->reject_reason",
                ];
                $user->notify(new UserNotice(subject: "Deposit $request->status", messages: $messages));
            }
        }

        return back()->with('success', $this->message('Deposit details updated.'));
    }

    public function delete(Deposit $deposit) {
        $deposit->delete();
        
        return to_route('admin.deposits.all')->with('success', $this->message('Deposit deleted successfully.'));
    }
}
